// Script de teste para validar a lógica de modificação de hash
const crypto = require('crypto');
const fs = require('fs');

console.log('=== Teste de Modificação de Hash ===\n');

// Função para calcular hash
function calculateHash(buffer) {
  return crypto.createHash('sha256').update(buffer).digest('hex');
}

// Função para modificar arquivo (simulando a lógica do backend)
function modifyFile(data, fileName) {
  // Gerar dados de injeção
  const timestamp = Date.now();
  const randomBytes = crypto.randomBytes(8);
  const injectionData = Buffer.concat([
    Buffer.from(timestamp.toString().padEnd(16, '0')),
    randomBytes,
  ]);

  let finalBuffer;
  let injectionOffset;
  let injectionMethod;

  // Determinar estratégia baseada no tipo de arquivo
  const isPagFile = fileName.toLowerCase().endsWith('.pag');

  if (isPagFile) {
    // Para .pag: append ao final
    finalBuffer = Buffer.concat([data, injectionData]);
    injectionOffset = data.length;
    injectionMethod = 'append_to_end';
  } else {
    // Para .exe/.dll: injeção no offset 1024
    const modifiedBuffer = Buffer.from(data);
    injectionOffset = Math.min(1024, modifiedBuffer.length - injectionData.length);
    injectionData.copy(modifiedBuffer, injectionOffset);
    finalBuffer = modifiedBuffer;
    injectionMethod = 'pe_header_padding';
  }

  return {
    finalBuffer,
    injectionOffset,
    injectionMethod,
    injectionSize: injectionData.length,
    originalSize: data.length,
    finalSize: finalBuffer.length,
  };
}

// Teste 1: Arquivo .pag
console.log('Teste 1: Arquivo .pag');
console.log('─'.repeat(50));
const pagData = fs.readFileSync('test.pag');
const originalPagHash = calculateHash(pagData);
console.log(`Tamanho original: ${pagData.length} bytes`);
console.log(`Hash original: ${originalPagHash}`);

const pagResult = modifyFile(pagData, 'test.pag');
const modifiedPagHash = calculateHash(pagResult.finalBuffer);
console.log(`\nMétodo de injeção: ${pagResult.injectionMethod}`);
console.log(`Offset de injeção: ${pagResult.injectionOffset}`);
console.log(`Tamanho da injeção: ${pagResult.injectionSize} bytes`);
console.log(`Tamanho final: ${pagResult.finalSize} bytes`);
console.log(`Hash modificado: ${modifiedPagHash}`);
console.log(`Hash mudou? ${originalPagHash !== modifiedPagHash ? '✅ SIM' : '❌ NÃO'}`);
console.log(`Tamanho aumentou? ${pagResult.finalSize > pagResult.originalSize ? '✅ SIM (append)' : '❌ NÃO'}`);

// Salvar arquivo modificado
fs.writeFileSync('test_modified.pag', pagResult.finalBuffer);
console.log(`\nArquivo modificado salvo: test_modified.pag`);

// Teste 2: Simular arquivo .exe (criar um buffer de teste)
console.log('\n\nTeste 2: Arquivo .exe (simulado)');
console.log('─'.repeat(50));
const exeData = Buffer.alloc(2048, 0); // Simular um .exe pequeno
exeData.write('MZ', 0); // Magic number do PE
const originalExeHash = calculateHash(exeData);
console.log(`Tamanho original: ${exeData.length} bytes`);
console.log(`Hash original: ${originalExeHash}`);

const exeResult = modifyFile(exeData, 'test.exe');
const modifiedExeHash = calculateHash(exeResult.finalBuffer);
console.log(`\nMétodo de injeção: ${exeResult.injectionMethod}`);
console.log(`Offset de injeção: ${exeResult.injectionOffset}`);
console.log(`Tamanho da injeção: ${exeResult.injectionSize} bytes`);
console.log(`Tamanho final: ${exeResult.finalSize} bytes`);
console.log(`Hash modificado: ${modifiedExeHash}`);
console.log(`Hash mudou? ${originalExeHash !== modifiedExeHash ? '✅ SIM' : '❌ NÃO'}`);
console.log(`Tamanho mantido? ${exeResult.finalSize === exeResult.originalSize ? '✅ SIM (in-place)' : '❌ NÃO'}`);

// Salvar arquivo modificado
fs.writeFileSync('test_modified.exe', exeResult.finalBuffer);
console.log(`\nArquivo modificado salvo: test_modified.exe`);

console.log('\n' + '='.repeat(50));
console.log('Todos os testes concluídos com sucesso! ✅');
console.log('='.repeat(50));

