import { COOKIE_NAME } from "@shared/const";
import { getSessionCookieOptions } from "./_core/cookies";
import { systemRouter } from "./_core/systemRouter";
import { publicProcedure, router } from "./_core/trpc";
import { z } from "zod";
import crypto from "crypto";
import fs from "fs";
import path from "path";
import os from "os";

export const appRouter = router({
  system: systemRouter,

  auth: router({
    me: publicProcedure.query(opts => opts.ctx.user),
    logout: publicProcedure.mutation(({ ctx }) => {
      const cookieOptions = getSessionCookieOptions(ctx.req);
      ctx.res.clearCookie(COOKIE_NAME, { ...cookieOptions, maxAge: -1 });
      return {
        success: true,
      } as const;
    }),
  }),

  // EXE/DLL/PAG Hash Modifier router
  exeModifier: router({
    // Calculate hash of a file buffer
    calculateHash: publicProcedure
      .input(z.object({ data: z.instanceof(Buffer) }))
      .query(({ input }) => {
        const hash = crypto.createHash('sha256').update(input.data).digest('hex');
        return { hash };
      }),

    // Modify file by injecting a timestamp at a safe offset
    // Supports .exe, .dll (PE format) and .pag files
    modifyExe: publicProcedure
      .input(z.object({ 
        data: z.instanceof(Buffer),
        fileName: z.string().optional()
      }))
      .mutation(async ({ input }) => {
        try {
          // Generate a unique injection: timestamp + random bytes
          const timestamp = Date.now();
          const randomBytes = crypto.randomBytes(8);
          const injectionData = Buffer.concat([
            Buffer.from(timestamp.toString().padEnd(16, '0')),
            randomBytes,
          ]);

          let finalBuffer: Buffer;
          let injectionOffset: number;
          let injectionMethod: string;

          // Determine file type and injection strategy
          const fileName = (input.fileName || '').toLowerCase();
          const isPagFile = fileName.endsWith('.pag');

          if (isPagFile) {
            // For .pag files: append data at the end (safest approach for unknown formats)
            // This creates a "padding area" at the end without corrupting the file structure
            finalBuffer = Buffer.concat([input.data, injectionData]);
            injectionOffset = input.data.length;
            injectionMethod = 'append_to_end';
          } else {
            // For .exe and .dll: inject at offset 1024 (safe PE header padding area)
            // PE files have padding in the header that can be safely modified
            const modifiedBuffer = Buffer.from(input.data);
            injectionOffset = Math.min(1024, modifiedBuffer.length - injectionData.length);
            injectionData.copy(modifiedBuffer, injectionOffset);
            finalBuffer = modifiedBuffer;
            injectionMethod = 'pe_header_padding';
          }

          // Calculate the new hash
          const newHash = crypto.createHash('sha256').update(finalBuffer).digest('hex');

          return {
            success: true,
            modifiedData: finalBuffer,
            newHash,
            injectionOffset,
            injectionSize: injectionData.length,
            injectionMethod,
            originalSize: input.data.length,
            finalSize: finalBuffer.length,
          };
        } catch (error) {
          throw new Error(`Failed to modify file: ${error instanceof Error ? error.message : String(error)}`);
        }
      }),
  }),
});

export type AppRouter = typeof appRouter;

