import { useState, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Upload, Download, Copy, CheckCircle2, AlertCircle } from "lucide-react";
import { APP_TITLE } from "@/const";

export default function Home() {
  const [file, setFile] = useState<File | null>(null);
  const [originalHash, setOriginalHash] = useState<string>("");
  const [modifiedHash, setModifiedHash] = useState<string>("");
  const [modifiedFile, setModifiedFile] = useState<Blob | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string>("");
  const [success, setSuccess] = useState(false);
  const [copiedField, setCopiedField] = useState<string | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const calculateHash = async (data: ArrayBuffer): Promise<string> => {
    const hashBuffer = await crypto.subtle.digest("SHA-256", data);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map((b) => b.toString(16).padStart(2, "0")).join("");
  };

  const handleFileSelect = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (!selectedFile) return;

    // Validate file type (.exe, .dll, or .pag)
    const fileName = selectedFile.name.toLowerCase();
    if (!fileName.endsWith(".exe") && !fileName.endsWith(".dll") && !fileName.endsWith(".pag")) {
      setError("Por favor, selecione um arquivo .exe, .dll ou .pag válido");
      return;
    }

    setFile(selectedFile);
    setError("");
    setSuccess(false);
    setModifiedHash("");
    setModifiedFile(null);

    // Calculate original hash
    try {
      const arrayBuffer = await selectedFile.arrayBuffer();
      const hash = await calculateHash(arrayBuffer);
      setOriginalHash(hash);
    } catch (err) {
      setError(`Erro ao calcular hash: ${err instanceof Error ? err.message : "Erro desconhecido"}`);
    }
  };

  const handleModifyExe = async () => {
    if (!file) {
      setError("Por favor, selecione um arquivo primeiro");
      return;
    }

    setLoading(true);
    setError("");

    try {
      const arrayBuffer = await file.arrayBuffer();
      const buffer = new Uint8Array(arrayBuffer);

      // Create a copy of the buffer
      const modifiedBuffer = new Uint8Array(buffer.length);
      modifiedBuffer.set(buffer);

      // Generate injection data: timestamp + random bytes
      const timestamp = Date.now();
      const randomBytes = crypto.getRandomValues(new Uint8Array(8));
      const timestampStr = timestamp.toString().padEnd(16, "0");
      const injectionData = new Uint8Array(timestampStr.length + randomBytes.length);

      // Copy timestamp string as bytes
      for (let i = 0; i < timestampStr.length; i++) {
        injectionData[i] = timestampStr.charCodeAt(i);
      }
      // Copy random bytes
      injectionData.set(randomBytes, timestampStr.length);

      // Determine injection strategy based on file type
      const fileName = file.name.toLowerCase();
      let finalBuffer: Uint8Array;
      let injectionOffset: number;

      if (fileName.endsWith(".pag")) {
        // For .pag files: append data at the end (safest approach)
        finalBuffer = new Uint8Array(modifiedBuffer.length + injectionData.length);
        finalBuffer.set(modifiedBuffer);
        finalBuffer.set(injectionData, modifiedBuffer.length);
        injectionOffset = modifiedBuffer.length;
      } else {
        // For .exe and .dll: inject at offset 1024 (safe PE header padding area)
        injectionOffset = Math.min(1024, modifiedBuffer.length - injectionData.length);
        modifiedBuffer.set(injectionData, injectionOffset);
        finalBuffer = modifiedBuffer;
      }

      // Calculate new hash
      const newHash = await calculateHash(finalBuffer.buffer);
      setModifiedHash(newHash);

      // Create blob for download
      const blob = new Blob([finalBuffer], { type: "application/octet-stream" });
      setModifiedFile(blob);
      setSuccess(true);
    } catch (err) {
      setError(`Erro ao modificar arquivo: ${err instanceof Error ? err.message : "Erro desconhecido"}`);
    } finally {
      setLoading(false);
    }
  };

  const handleDownload = () => {
    if (!modifiedFile || !file) return;

    const url = URL.createObjectURL(modifiedFile);
    const link = document.createElement("a");
    link.href = url;
    link.download = file.name;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
  };

  const copyToClipboard = (text: string, field: string) => {
    navigator.clipboard.writeText(text);
    setCopiedField(field);
    setTimeout(() => setCopiedField(null), 2000);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-slate-100 dark:from-slate-950 dark:to-slate-900 py-8 px-4">
      <div className="max-w-2xl mx-auto">
        {/* Header */}
        <div className="text-center mb-8">
          <h1 className="text-4xl font-bold text-slate-900 dark:text-white mb-2">{APP_TITLE}</h1>
          <p className="text-lg text-slate-600 dark:text-slate-400">
            Altere o hash de seus arquivos .exe, .dll ou .pag com segurança
          </p>
        </div>

        {/* Main Card */}
        <Card className="shadow-lg">
          <CardHeader className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-950 dark:to-indigo-950">
            <CardTitle>Modificador de Hash PE (EXE/DLL/PAG)</CardTitle>
            <CardDescription>
              Faça upload de um arquivo .exe, .dll ou .pag para alterar seu hash mantendo a funcionalidade intacta
            </CardDescription>
          </CardHeader>

          <CardContent className="pt-6 space-y-6">
            {/* File Upload Section */}
            <div className="space-y-4">
              <div
                className="border-2 border-dashed border-slate-300 dark:border-slate-600 rounded-lg p-8 text-center cursor-pointer hover:border-blue-500 dark:hover:border-blue-400 transition-colors"
                onClick={() => fileInputRef.current?.click()}
              >
              <input
                ref={fileInputRef}
                type="file"
                accept=".exe,.dll,.pag"
                onChange={handleFileSelect}
                className="hidden"
              />
                <Upload className="w-12 h-12 mx-auto mb-3 text-slate-400" />
                <p className="text-sm font-medium text-slate-700 dark:text-slate-300">
                  {file ? file.name : "Clique para selecionar um arquivo .exe, .dll ou .pag"}
                </p>
                <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">
                  ou arraste e solte aqui
                </p>
              </div>

              {/* Error Message */}
              {error && (
                <div className="flex items-center gap-2 p-4 bg-red-50 dark:bg-red-950 border border-red-200 dark:border-red-800 rounded-lg">
                  <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400" />
                  <p className="text-sm text-red-700 dark:text-red-300">{error}</p>
                </div>
              )}

              {/* Success Message */}
              {success && (
                <div className="flex items-center gap-2 p-4 bg-green-50 dark:bg-green-950 border border-green-200 dark:border-green-800 rounded-lg">
                  <CheckCircle2 className="w-5 h-5 text-green-600 dark:text-green-400" />
                  <p className="text-sm text-green-700 dark:text-green-300">
                    Arquivo modificado com sucesso! Hash alterado.
                  </p>
                </div>
              )}
            </div>

            {/* Hash Display Section */}
            {file && (
              <div className="space-y-4">
                {/* Original Hash */}
                <div className="space-y-2">
                  <label className="block text-sm font-medium text-slate-700 dark:text-slate-300">
                    Hash SHA256 Original
                  </label>
                  <div className="flex items-center gap-2">
                    <code className="flex-1 p-3 bg-slate-100 dark:bg-slate-800 rounded text-xs font-mono text-slate-700 dark:text-slate-300 break-all">
                      {originalHash || "Calculando..."}
                    </code>
                    {originalHash && (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => copyToClipboard(originalHash, "original")}
                        className="flex-shrink-0"
                      >
                        {copiedField === "original" ? "✓" : <Copy className="w-4 h-4" />}
                      </Button>
                    )}
                  </div>
                </div>

                {/* Modify Button */}
                <Button
                  onClick={handleModifyExe}
                  disabled={loading || !originalHash}
                  className="w-full bg-blue-600 hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-600"
                  size="lg"
                >
                  {loading ? "Modificando..." : "Modificar Hash"}
                </Button>

                {/* Modified Hash */}
                {modifiedHash && (
                  <div className="space-y-2">
                    <label className="block text-sm font-medium text-slate-700 dark:text-slate-300">
                      Hash SHA256 Modificado
                    </label>
                    <div className="flex items-center gap-2">
                      <code className="flex-1 p-3 bg-slate-100 dark:bg-slate-800 rounded text-xs font-mono text-slate-700 dark:text-slate-300 break-all">
                        {modifiedHash}
                      </code>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => copyToClipboard(modifiedHash, "modified")}
                        className="flex-shrink-0"
                      >
                        {copiedField === "modified" ? "✓" : <Copy className="w-4 h-4" />}
                      </Button>
                    </div>
                  </div>
                )}

                {/* Download Button */}
                {modifiedFile && (
                  <Button
                    onClick={handleDownload}
                    className="w-full bg-green-600 hover:bg-green-700 dark:bg-green-700 dark:hover:bg-green-600"
                    size="lg"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Baixar Arquivo Modificado
                  </Button>
                )}
              </div>
            )}

            {/* Info Section */}
            <div className="bg-blue-50 dark:bg-blue-950 border border-blue-200 dark:border-blue-800 rounded-lg p-4 space-y-2">
              <h3 className="font-semibold text-blue-900 dark:text-blue-100">Como funciona?</h3>
              <ul className="text-sm text-blue-800 dark:text-blue-200 space-y-1">
                <li>• Suporta arquivos .exe, .dll (formato PE) e .pag</li>
                <li>• Injetamos um timestamp único em área segura do arquivo</li>
                <li>• Para .exe/.dll: injeção no padding do header PE (offset 1024)</li>
                <li>• Para .pag: injeção ao final do arquivo (append)</li>
                <li>• O hash SHA256 muda completamente sem corromper a estrutura</li>
                <li>• A funcionalidade do arquivo permanece intacta</li>
                <li>• Cada modificação gera um hash único</li>
              </ul>
            </div>
          </CardContent>
        </Card>

        {/* Footer */}
        <div className="text-center mt-8 text-sm text-slate-600 dark:text-slate-400">
          <p>Ferramenta segura e rápida para modificação de hash de executáveis (.exe), bibliotecas (.dll) e arquivos .pag</p>
        </div>
      </div>
    </div>
  );
}

